let player;
let timelineInterval;
let interactionsData = [];
let currentLiveEditIndex = null;
let currentTimeoutId = null;
let desiredStartTime = null;

function getYoutubeVideoId(url) {
    if (!url) return null;
    // CORRIGÉ : La regex accepte maintenant correctement les URL "watch?v="
    const regExp = /^.*(youtu.be\/|v\/|u\/\w\/|embed\/|watch\?v=|&v=)([^#&?]*).*/;
    const match = url.match(regExp);
    return (match && match[2].length === 11) ? match[2] : null;
}

function showLiveEditor(interaction, index) {
    currentLiveEditIndex = index;
    const container = document.getElementById('live-editor-container');
    const contentEl = document.getElementById('live-editor-content');
    const choiceData = interaction.action.params.choices[0];
    let answersHtml = '';
    choiceData.answers.forEach((answer, aIndex) => {
        answersHtml += `<div class="answer-group">
            <input type="radio" name="live-correct-answer" value="${aIndex}" ${aIndex === 0 ? 'checked' : ''}>
            <textarea class="answer-input" rows="1" data-a-index="${aIndex}">${answer.replace(/<\/?p>/g, '')}</textarea>
        </div>`;
    });

    contentEl.innerHTML = `
        <div class="form-group-editor"><label>Texte de la question</label><textarea id="live-question-text">${choiceData.question.replace(/<\/?p>/g, '')}</textarea></div>
        <div class="form-group-editor"><label>Choix de réponse</label>${answersHtml}</div>
        <div class="form-grid-editor">
            <div class="form-group-editor"><label>Temps d'apparition (sec)</label><input type="number" id="live-appearance-time" value="${interaction.duration.to}"></div>
            <div class="form-group-editor"><label>Retour en arrière (sec)</label><input type="number" id="live-seek-back-time" value="${interaction.adaptivity.wrong.seek}"></div>
        </div>
        <div class="form-group-editor"><label>Rétroaction</label><textarea id="live-feedback">${interaction.action.params.overallFeedback[0].feedback}</textarea></div>
    `;
    container.classList.remove('hidden');

    function autoResizeTextarea(event) {
        event.target.style.height = 'auto';
        event.target.style.height = (event.target.scrollHeight) + 'px';
    }
    contentEl.querySelectorAll('textarea.answer-input').forEach(textarea => {
        textarea.addEventListener('input', autoResizeTextarea);
        textarea.style.height = 'auto';
        textarea.style.height = (textarea.scrollHeight) + 'px';
    });
}

function hideLiveEditor() {
    currentLiveEditIndex = null;
    document.getElementById('live-editor-container').classList.add('hidden');
}

function resetModal(submitHandler) {
    const answersEl = document.getElementById('question-answers');
    const feedbackEl = document.getElementById('question-feedback');
    const submitBtn = document.getElementById('question-submit-btn');
    answersEl.classList.remove('answered');
    feedbackEl.classList.add('hidden');
    submitBtn.textContent = 'Valider la réponse';
    submitBtn.disabled = false;
    answersEl.querySelectorAll('.answer-label').forEach(label => {
        label.classList.remove('correct', 'incorrect');
        label.querySelector('input').checked = false;
    });
    if (submitHandler) {
        submitBtn.onclick = submitHandler;
    }
}

function showQuestionModal(interaction, index) {
    showLiveEditor(interaction, index);
    
    const modal = document.getElementById('question-modal-overlay');
    const questionEl = document.getElementById('question-text');
    const answersEl = document.getElementById('question-answers');
    const feedbackEl = document.getElementById('question-feedback');
    
    const modalContent = document.getElementById('question-modal-content');
    modalContent.style.position = 'relative'; 

    const skipQuestion = () => {
        modal.classList.add('hidden');
        hideLiveEditor();
        document.body.classList.remove('modal-open');
        player.playVideo();
    };

    let skipBtn = modalContent.querySelector('.preview-skip-btn');
    if (!skipBtn) {
        skipBtn = document.createElement('button');
        skipBtn.innerHTML = '&times;';
        skipBtn.title = 'Passer la question';
        skipBtn.className = 'preview-skip-btn';
        Object.assign(skipBtn.style, {
            position: 'absolute', top: '10px', right: '15px',
            background: 'none', border: 'none', fontSize: '2em',
            lineHeight: '1', cursor: 'pointer', padding: '0', color: '#888'
        });
        modalContent.appendChild(skipBtn);
    }
    skipBtn.onclick = skipQuestion;
    
    const behaviour = interaction.action.params.behaviour ?? {};
    const adaptivity = interaction.adaptivity ?? {};
    const autoContinue = behaviour.autoContinue ?? true;
    const enableRetry = behaviour.enableRetry ?? true;
    const timeoutCorrect = behaviour.timeoutCorrect ?? 2000;
    const timeoutWrong = behaviour.timeoutWrong ?? 3000;
    const seekBackTime = adaptivity.wrong?.seek;
    const feedbackText = interaction.action.params.overallFeedback[0].feedback;

    const continueAction = (isCorrect) => {
        modal.classList.add('hidden');
        hideLiveEditor();
        document.body.classList.remove('modal-open');
        if (!isCorrect && seekBackTime !== undefined) {
            player.seekTo(seekBackTime, true);
        }
        player.playVideo();
    };

    const handleSubmit = () => {
        const submitBtn = document.getElementById('question-submit-btn');
        const selectedRadio = answersEl.querySelector('input[name="preview-answer"]:checked');
        if (!selectedRadio) { alert("Veuillez sélectionner une réponse."); return; }

        answersEl.classList.add('answered');
        const selectedIndex = parseInt(selectedRadio.value, 10);
        const isCorrect = selectedIndex === 0;

        if (feedbackText && feedbackText.trim() !== '') {
            feedbackEl.innerHTML = feedbackText;
            if(window.MathJax) MathJax.typesetPromise([feedbackEl]);
            feedbackEl.classList.remove('hidden');
        }
        
        selectedRadio.parentElement.classList.add(isCorrect ? 'correct' : 'incorrect');
        if (!isCorrect) {
            answersEl.querySelector('input[value="0"]').parentElement.classList.add('correct');
        }

        if (isCorrect) {
            if (autoContinue) {
                submitBtn.textContent = "Correct !";
                submitBtn.disabled = true;
                currentTimeoutId = setTimeout(() => continueAction(true), timeoutCorrect);
            } else {
                submitBtn.textContent = 'Continuer';
                submitBtn.onclick = () => continueAction(true);
            }
        } else {
            if (enableRetry) {
                if (autoContinue) {
                    submitBtn.textContent = "Incorrect...";
                    submitBtn.disabled = true;
                    currentTimeoutId = setTimeout(() => continueAction(false), timeoutWrong);
                } else {
                    submitBtn.textContent = 'Continuer';
                    submitBtn.onclick = () => continueAction(false);
                }
            } else {
                submitBtn.textContent = 'Réessayer';
                submitBtn.onclick = () => resetModal(handleSubmit);
            }
        }
    };

    resetModal(handleSubmit);
    const choiceData = interaction.action.params.choices[0];
    questionEl.innerHTML = choiceData.question;
    answersEl.innerHTML = '';
    choiceData.answers.forEach((answer, idx) => {
        answersEl.innerHTML += `<label class="answer-label"><input type="radio" name="preview-answer" value="${idx}"><span>${answer.replace(/<\/?p>/g, '')}</span></label>`;
    });

    if (window.MathJax) {
        MathJax.typesetPromise([modal]);
    }
    
    document.body.classList.add('modal-open');
    modal.classList.remove('hidden');
}

function handleSeek(newTime) {
    interactionsData.forEach(interaction => {
        if (interaction.duration.to > newTime) {
            interaction.triggered = false;
        }
    });
    player.seekTo(newTime, true);
}

function updateTimeline() {
    if (!player || typeof player.getCurrentTime !== 'function') return;
    const currentTime = player.getCurrentTime();
    const duration = player.getDuration();
    if (!duration) return;
    document.getElementById('video-timeline').value = currentTime;
    document.getElementById('video-timeline').max = duration;

    for (let i = 0; i < interactionsData.length; i++) {
        const interaction = interactionsData[i];
        if (currentTime >= interaction.duration.to && !interaction.triggered) {
            interaction.triggered = true;
            player.pauseVideo();
            showQuestionModal(interaction, i);
            break;
        }
    }
}

function renderTimelineMarkers() {
    const markersContainer = document.getElementById('timeline-markers');
    const duration = player.getDuration();
    if (!duration) return;
    markersContainer.innerHTML = '';
    interactionsData.forEach((interaction, index) => {
        const marker = document.createElement('div');
        marker.className = 'timeline-marker';
        const thumbWidth = 18;
        const percentage = interaction.duration.to / duration;
        marker.style.left = `calc(${percentage * 100}% - ${thumbWidth * percentage}px)`;
        marker.title = `Aller à la question (${Math.round(interaction.duration.to)}s)`;
        
        marker.addEventListener('click', (event) => {
            event.stopPropagation();
            
            document.getElementById('video-timeline').value = interaction.duration.to;
            
            interactionsData.forEach(i => {
                if (i.duration.to < interaction.duration.to) i.triggered = true;
            });
            interaction.triggered = true;

            player.seekTo(interaction.duration.to, true);
            player.pauseVideo();
            showQuestionModal(interaction, index);
        });
        
        markersContainer.appendChild(marker);
    });
}

export function initializePreview(config) {
    const messageContainer = document.getElementById('preview-message-container');
    const contentWrapper = document.getElementById('preview-content-wrapper');
    if (!config || !config.editorContent) {
        messageContainer.innerHTML = '<p>Chargez une configuration pour lancer la prévisualisation.</p>';
        messageContainer.classList.remove('hidden');
        contentWrapper.classList.add('hidden');
        return;
    }
    const videoId = getYoutubeVideoId(config.context.videoUrl);
    if (!videoId) {
        messageContainer.innerHTML = '<p>L\'URL de la vidéo YouTube est invalide ou manquante dans la configuration.</p>';
        messageContainer.classList.remove('hidden');
        contentWrapper.classList.add('hidden');
        return;
    }
    messageContainer.classList.add('hidden');
    contentWrapper.classList.remove('hidden');
    interactionsData = JSON.parse(JSON.stringify(config.editorContent.interactiveVideo.assets.interactions)).map(i => ({...i, triggered: false }));
    if (player && typeof player.destroy === 'function') {
        player.destroy();
    }
    player = new YT.Player('youtube-player-container', {
        height: '100%', width: '100%', videoId: videoId,
        playerVars: { 'playsinline': 1, 'controls': 0, 'rel': 0 },
        events: { 'onReady': onPlayerReady, 'onStateChange': onPlayerStateChange }
    });
    document.getElementById('video-timeline').addEventListener('input', (e) => {
        handleSeek(parseFloat(e.target.value));
    });
}

function onPlayerReady(event) {
    renderTimelineMarkers();
    
    if (desiredStartTime !== null) {
        interactionsData.forEach(interaction => {
            if (interaction.duration.to <= desiredStartTime) {
                interaction.triggered = true;
            }
        });

        event.target.seekTo(desiredStartTime, true);
        desiredStartTime = null;
    }
    event.target.playVideo();
}

function onPlayerStateChange(event) {
    clearInterval(timelineInterval);
    if (event.data === YT.PlayerState.PLAYING) {
        timelineInterval = setInterval(updateTimeline, 250);
    }
}

export function destroyPreview() {
    clearInterval(timelineInterval);
    clearTimeout(currentTimeoutId);
    hideLiveEditor();
    if (player && typeof player.destroy === 'function') {
        player.destroy();
    }
    player = null;
    document.getElementById('question-modal-overlay').classList.add('hidden');
    document.body.classList.remove('modal-open');
    const container = document.getElementById('youtube-player-wrapper');
    if (container) container.innerHTML = '<div id="youtube-player-container"></div>';
    document.getElementById('preview-message-container').innerHTML = '<p>La prévisualisation est terminée. Cliquez à nouveau sur l\'onglet pour la relancer.</p>';
    document.getElementById('preview-message-container').classList.remove('hidden');
    document.getElementById('preview-content-wrapper').classList.add('hidden');
}

export function getLiveEditorChanges() {
    if (currentLiveEditIndex === null) return null;

    const newAppearanceTime = parseInt(document.getElementById('live-appearance-time').value, 10);
    desiredStartTime = Math.max(0, newAppearanceTime - 5);

    const correctRadio = document.querySelector('input[name="live-correct-answer"]:checked');
    const correctIndexInUI = correctRadio ? parseInt(correctRadio.value, 10) : 0;
    const answerInputs = Array.from(document.querySelectorAll('#live-editor-container .answer-input'));
    let answersText = answerInputs.map(input => `<p>${input.value}</p>`);
    const correctAnswerText = answersText.splice(correctIndexInUI, 1)[0];
    answersText.unshift(correctAnswerText);
    const originalInteraction = interactionsData[currentLiveEditIndex];
    const updatedInteraction = { ...originalInteraction, duration: { ...originalInteraction.duration, to: newAppearanceTime }, action: { ...originalInteraction.action, params: { ...originalInteraction.action.params, choices: [{ question: `<p>${document.getElementById('live-question-text').value}</p>`, answers: answersText }], overallFeedback: [{ ...originalInteraction.action.params.overallFeedback[0], feedback: document.getElementById('live-feedback').value }] } }, adaptivity: { ...originalInteraction.adaptivity, wrong: { seek: parseInt(document.getElementById('live-seek-back-time').value, 10) } } };
    
    return { index: currentLiveEditIndex, updatedInteraction };
}